/*
 * Copyright (C) 2025 Hiroki Sato
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 3.
 *
 * This program uses Arduino SD library
 * Copyright (C) Yoshino Taro
 */

#include <Arduino.h>
#include <Camera.h>
#include <SPI.h>
#include <SPISD.h>
#include <LowPower.h>
#include <RTC.h>

/* =========================================================
   [U[ݒ
   ========================================================= */

/*
  BeԊui~bj
   - 10b   : 10000
   - 1    : 60000
   - 10   : 600000
*/
const uint32_t INTERVAL_SEC = 10;

/*
  }CNSDJ[hdsǂ
   true  : D25PMOSQ[g
   false : SDJ[h펞ʓd
*/
const bool USE_SD_POWER_CONTROL = false;

/*
  PMOS Q[gs
   H : SDJ[hd OFF
   L : SDJ[hd ON
*/
const int SD_POWER_PIN = 25;

/* ========================================================= */

/* SPI5 g SD NX */
SpiSDClass SD(SPI5);

/* BeJE^
   DeepSleep Ap邽 SD ɕۑ */
static uint32_t gCounter = 0;
static const char COUNTER_FILE[] = "count.txt";

/* =========================================================
   JE^ǂݍ/ۑ֐̐ݒ
   ========================================================= */
void loadCounter() {
  // JE^t@C 0 X^[g
  if (!SD.exists(COUNTER_FILE)) {
    gCounter = 0;
    return;
  }

  SpiFile f = SD.open(COUNTER_FILE, FILE_READ);
  if (!f) {
    gCounter = 0;
    return;
  }

  // 1sڂɏꂽlǂ
  gCounter = f.readStringUntil('\n').toInt();
  f.close();
}

void saveCounter() {
  // 񏑂iȒPŊmj
  if (SD.exists(COUNTER_FILE)) {
    SD.remove(COUNTER_FILE);
  }

  SpiFile f = SD.open(COUNTER_FILE, FILE_WRITE);
  if (!f) return;

  f.println(gCounter);
  f.close();
}

/* =========================================================
   setup()
   ========================================================= */

void setup() {
  Serial.begin(115200);
  while (!Serial);

  Serial.println("=== Spresense Timelapse DeepSleep ===");

  /* ---- LowPower / RTC  ---- */
  LowPower.begin();
  RTC.begin();

  /* ---- N@̔ ----
     DEEP_RTC : DeepSleep ̕A
     ȊO : dON / Zbg */
  bootcause_e bc = LowPower.bootCause();
  if (bc == DEEP_RTC) {
    Serial.println("Wakeup from DeepSleep");
  } else {
    Serial.println("Power On / Reset");
  }

  /* ---- SD dsݒ ---- */
  if (USE_SD_POWER_CONTROL) {
    pinMode(SD_POWER_PIN, OUTPUT);
    digitalWrite(SD_POWER_PIN, HIGH);   // SD d OFF
  }

  /* ---- SD d ON ---- */
  if (USE_SD_POWER_CONTROL) {
    digitalWrite(SD_POWER_PIN, LOW);
    delay(100); // d҂
  }

  /* ---- SD  ---- */
  Serial.print("Initializing SD... ");
  if (!SD.begin(SPI_FULL_SPEED)) {
    Serial.println("FAILED");
    while (1);
  }
  Serial.println("OK");

  /* ---- BeJE^ ---- */
  loadCounter();
  Serial.print("Frame Counter = ");
  Serial.println(gCounter);

  /* ---- SD d OFF ---- */
  if (USE_SD_POWER_CONTROL) {
    digitalWrite(SD_POWER_PIN, HIGH);
  }

  /* ---- Camera  ---- */
  Serial.print("Initializing Camera... ");
  CamErr err = theCamera.begin();
  if (err != CAM_ERR_SUCCESS) {
    Serial.print("FAILED: ");
    Serial.println(err);
    while (1);
  }
  Serial.println("OK");

  /* ---- Î~ JPEG ݒ ---- */
  theCamera.setStillPictureImageFormat(
    CAM_IMGSIZE_VGA_H,
    CAM_IMGSIZE_VGA_V,
    CAM_IMAGE_PIX_FMT_JPG
  );

  Serial.println("Setup done.");
}

/* =========================================================
   loop()
   ========================================================= */

void loop() {

  Serial.println("Taking picture...");

  /* ---- Be ---- */
  CamImage img = theCamera.takePicture();
  if (!img.isAvailable()) {
    Serial.println("Capture FAILED!");
    theCamera.end();
    LowPower.deepSleep(INTERVAL_SEC);
    return;
  }

  /* ---- t@C ---- */
  char filename[16];
  sprintf(filename, "PICT%04lu.JPG", gCounter);

  /* ---- SD d ON ---- */
  if (USE_SD_POWER_CONTROL) {
    digitalWrite(SD_POWER_PIN, LOW);
    delay(100);
  }

  /* ---- SD  ---- */
  SpiFile f = SD.open(filename, FILE_WRITE);
  if (!f) {
    Serial.println("File open FAILED");
    if (USE_SD_POWER_CONTROL) {
      digitalWrite(SD_POWER_PIN, HIGH);
    }
    theCamera.end();
    LowPower.deepSleep(INTERVAL_SEC);
    return;
  }

  f.write(img.getImgBuff(), img.getImgSize());
  f.close();

  /* ---- JE^XV ---- */
  gCounter++;
  saveCounter();

  /* ---- SD d OFF ---- */
  if (USE_SD_POWER_CONTROL) {
    digitalWrite(SD_POWER_PIN, HIGH);
  }

  Serial.print("Saved: ");
  Serial.println(filename);

  /* ---- J~ ---- */
  theCamera.end();

  /* ---- DeepSleep ---- */
  Serial.println("Go to DeepSleep...");
  LowPower.deepSleep(INTERVAL_SEC);
}
